// Pingbird Prompt Compressor - Browser Extension Version
// All processing happens locally in your browser

const promptPreambles = {
  "you are chatgpt, a large language model trained by openai": "You are ChatGPT (OpenAI LLM)",
  "you are chatgpt a large language model trained by openai": "You are ChatGPT (OpenAI LLM)",
  "you are gpt-4, a large language model trained by openai": "You are GPT-4 (OpenAI LLM)",
  "you are gpt-4o, a large language model trained by openai": "You are GPT-4o (OpenAI LLM)",
  "your goal is to": "Goal:",
  "your goals are to": "Goals:",
  "your objective is to": "Objective:",
  "your objectives are to": "Objectives:",
  "your role is to": "Role:",
  "your task is to": "Task:",
};

function matchReplacementCase(source, replacement) {
  if (!source) return replacement;

  const letters = source.replace(/[^a-z]/gi, "");
  if (letters) {
    const isAllUppercase = letters === letters.toUpperCase();
    if (isAllUppercase && letters !== letters.toLowerCase()) {
      return replacement.toUpperCase();
    }
  }

  const leading = source.trimStart()[0];
  if (!leading || leading.toLowerCase() === leading.toUpperCase()) {
    return replacement;
  }

  if (leading === leading.toUpperCase() && leading !== leading.toLowerCase()) {
    if (!replacement) return replacement;
    return replacement[0].toUpperCase() + replacement.slice(1);
  }

  if (leading === leading.toLowerCase() && leading !== leading.toUpperCase()) {
    if (!replacement) return replacement;
    return replacement[0].toLowerCase() + replacement.slice(1);
  }

  return replacement;
}

function compressPromptPreamble(text) {
  if (!text) return text;

  const leadingWhitespaceMatch = text.match(/^\s*/);
  const leadingWhitespace = leadingWhitespaceMatch ? leadingWhitespaceMatch[0] : "";
  const remainder = text.slice(leadingWhitespace.length);

  for (const [preamble, shortened] of Object.entries(promptPreambles)) {
    if (remainder.toLowerCase().startsWith(preamble)) {
      const matchedSegment = remainder.slice(0, preamble.length);
      const adjusted = matchReplacementCase(matchedSegment, shortened);
      return `${leadingWhitespace}${adjusted}${remainder.slice(preamble.length)}`;
    }
  }

  return text;
}

const abbreviations = {
  application: "app",
  applications: "apps",
  information: "info",
  configuration: "config",
  administrator: "admin",
  development: "dev",
  production: "prod",
  environment: "env",
  database: "db",
  maximum: "max",
  minimum: "min",
  number: "num",
  document: "doc",
  documents: "docs",
  documentation: "docs",
  example: "eg",
  examples: "eg",
  etcetera: "etc",
  versus: "vs",
  approximate: "approx",
  approximately: "~",
  temperature: "temp",
  message: "msg",
  messages: "msgs",
  response: "resp",
  request: "req",
  parameter: "param",
  parameters: "params",
  argument: "arg",
  arguments: "args",
  function: "func",
  functions: "funcs",
  variable: "var",
  variables: "vars",
  constant: "const",
  constants: "consts",
  reference: "ref",
  references: "refs",
  definition: "def",
  definitions: "defs",
  specification: "spec",
  specifications: "specs",
  authentication: "auth",
  authorization: "authz",
  implementation: "impl",
  implementations: "impls",
  repository: "repo",
  repositories: "repos",
  directory: "dir",
  directories: "dirs",
  temporary: "temp",
  permanent: "perm",
  description: "desc",
  descriptions: "descs",
  management: "mgmt",
  manager: "mgr",
  process: "proc",
  processes: "procs",
  system: "sys",
  systems: "sys",
  command: "cmd",
  commands: "cmds",
  utility: "util",
  utilities: "utils",
  version: "ver",
  versions: "vers",
  you: "u",
  your: "ur",
  yours: "urs",
  yourself: "urself",
  yourselves: "urselves",
};

const apostropheSensitiveAbbreviations = new Set([
  "you",
  "your",
  "yours",
  "yourself",
  "yourselves",
]);

const phraseReplacements = {
  "in order to": "to",
  "as a result of": "due to",
  "at this point in time": "now",
  "at the present time": "now",
  "in the event that": "if",
  "with regard to": "about",
  "with respect to": "about",
  "in regards to": "about",
  "for the purpose of": "for",
  "in the process of": "while",
  "make use of": "use",
  "make sure that": "ensure",
  "please make sure": "ensure",
  "it is important to": "must",
  "it is necessary to": "must",
  "you should": "must",
  "you need to": "must",
  "you must": "must",
  "do not": "don't",
  "will not": "won't",
  cannot: "can't",
  "should not": "shouldn't",
  "could not": "couldn't",
  "would not": "wouldn't",
  "is not": "isn't",
  "are not": "aren't",
  "was not": "wasn't",
  "were not": "weren't",
  "has not": "hasn't",
  "have not": "haven't",
  "had not": "hadn't",
  "a large number of": "many",
  "a small number of": "few",
  "the majority of": "most",
  "each and every": "each",
  "first and foremost": "first",
  "in close proximity": "near",
  "in my opinion": "",
  "i think that": "",
  "it seems that": "",
  "there is": "there's",
  "there are": "there're",
  "that is": "that's",
  "which is": "which's",
  "who is": "who's",
  "what is": "what's",
  "where is": "where's",
  "when is": "when's",
  "how is": "how's",
  "here is": "here's",
  "and so on": "etc",
  "and so forth": "etc",
  "at all times": "always",
  "on a daily basis": "daily",
  "on a weekly basis": "weekly",
  "on a monthly basis": "monthly",
  "take into consideration": "consider",
  "give consideration to": "consider",
  "come to a conclusion": "conclude",
  "make a decision": "decide",
  "conduct an investigation": "investigate",
  "perform an analysis": "analyze",
  "in addition to": "plus",
  "as well as": "and",
  "whether or not": "whether",
};

const fillerWords = new Set(
  [
    "actually",
    "basically",
    "certainly",
    "definitely",
    "essentially",
    "extremely",
    "fairly",
    "highly",
    "just",
    "quite",
    "rather",
    "really",
    "simply",
    "somewhat",
    "totally",
    "utterly",
    "very",
    "pretty",
    "perhaps",
    "possibly",
    "probably",
    "maybe",
    "clearly",
    "obviously",
    "apparently",
    "seemingly",
    "evidently",
    "supposedly",
    "arguably",
    "literally",
    "virtually",
    "practically",
    "nearly",
  ].map((word) => word.toLowerCase())
);

const redundantArticles = new Set(
  ["the", "a", "an", "this", "that", "these", "those", "some", "any", "much", "many", "few", "several"].map((word) =>
    word.toLowerCase()
  )
);

function escapeRegExp(value) {
  return value.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");
}

function preserveCase(example, replacement) {
  const alpha = example.replace(/[^a-z]/gi, "");
  if (!alpha) return replacement;
  if (alpha === alpha.toUpperCase()) return replacement.toUpperCase();
  const first = alpha[0];
  if (first && first === first.toUpperCase()) {
    return replacement.charAt(0).toUpperCase() + replacement.slice(1);
  }
  return replacement;
}

const quantityUnits = {
  millisecond: "ms",
  milliseconds: "ms",
  second: "s",
  seconds: "s",
  minute: "min",
  minutes: "min",
  hour: "h",
  hours: "h",
  day: "d",
  days: "d",
  byte: "B",
  bytes: "B",
  kilobyte: "KB",
  kilobytes: "KB",
  megabyte: "MB",
  megabytes: "MB",
  percent: "%",
  percentage: "%",
  degree: "deg",
  degrees: "deg",
};

const numberWordValues = {
  zero: 0,
  one: 1,
  two: 2,
  three: 3,
  four: 4,
  five: 5,
  six: 6,
  seven: 7,
  eight: 8,
  nine: 9,
  ten: 10,
  eleven: 11,
  twelve: 12,
  thirteen: 13,
  fourteen: 14,
  fifteen: 15,
  sixteen: 16,
  seventeen: 17,
  eighteen: 18,
  nineteen: 19,
};

const tensWordValues = {
  twenty: 20,
  thirty: 30,
  forty: 40,
  fifty: 50,
  sixty: 60,
  seventy: 70,
  eighty: 80,
  ninety: 90,
};

const numberWordTokens = new Set([
  ...Object.keys(numberWordValues),
  ...Object.keys(tensWordValues),
  "hundred",
  "thousand",
  "and",
]);

const numberWordPattern = new RegExp(
  `\\b(?:${Array.from(numberWordTokens).join("|")})(?:[-\\s]+(?:${Array.from(numberWordTokens).join("|")}))*\\b`,
  "gi"
);

function parseNumberWords(input) {
  const tokens = input
    .toLowerCase()
    .split(/[-\s]+/)
    .filter(Boolean);

  let total = 0;
  let current = 0;
  let matched = false;

  for (const token of tokens) {
    if (token === "and") continue;
    if (token === "hundred") {
      matched = true;
      current = (current === 0 ? 1 : current) * 100;
      continue;
    }
    if (token === "thousand") {
      matched = true;
      total += (current === 0 ? 1 : current) * 1000;
      current = 0;
      continue;
    }
    if (token in numberWordValues) {
      matched = true;
      current += numberWordValues[token];
      continue;
    }
    if (token in tensWordValues) {
      matched = true;
      current += tensWordValues[token];
      continue;
    }
    return null;
  }

  if (!matched) return null;
  return total + current;
}

function compressQuantities(text) {
  let output = text.replace(numberWordPattern, (match) => {
    const numeric = parseNumberWords(match);
    return numeric === null ? match : String(numeric);
  });

  output = Object.entries(quantityUnits).reduce((acc, [from, to]) => {
    const pattern = new RegExp(`\\b${escapeRegExp(from)}\\b`, "gi");
    return acc.replace(pattern, to);
  }, output);

  return output;
}

function applyVerbosePhraseCompression(text) {
  return Object.entries(phraseReplacements).reduce((acc, [from, to]) => {
    const pattern = new RegExp(`\\b${escapeRegExp(from)}\\b`, "gi");
    return acc.replace(pattern, (match) => preserveCase(match, to));
  }, text);
}

function applyAbbreviationCompression(text) {
  return Object.entries(abbreviations).reduce((acc, [from, to]) => {
    const guard = apostropheSensitiveAbbreviations.has(from) ? "(?![''])" : "";
    const pattern = new RegExp(`\\b${escapeRegExp(from)}\\b${guard}`, "gi");
    return acc.replace(pattern, (match) => preserveCase(match, to));
  }, text);
}

function removeRepeatedFillerWords(text) {
  const counts = new Map();
  return text.replace(/\b[\p{L}']+\b/giu, (match) => {
    const canonical = match.toLowerCase();
    if (!fillerWords.has(canonical)) return match;
    const seen = counts.get(canonical) ?? 0;
    counts.set(canonical, seen + 1);
    return seen === 0 ? match : "";
  });
}

function trimRedundantArticles(text) {
  return text.replace(/([^.!?]*)([.!?]*)/g, (segment, clause, punctuation) => {
    if (!clause) return segment;
    let seenArticle = false;
    const processed = clause.replace(/\b[\p{L}']+\b/giu, (match) => {
      const canonical = match.toLowerCase();
      if (!redundantArticles.has(canonical)) return match;
      if (!seenArticle) {
        seenArticle = true;
        return match;
      }
      return "";
    });
    return `${processed}${punctuation}`;
  });
}

function preferImperativeTone(text) {
  return text
    .replace(/\b(please|kindly|try to|make sure to|be sure to|remember to|don't forget to)\b/gi, "")
    .replace(/\b(you should|you need to|you must)\b/gi, "")
    .replace(/\s+/g, " ")
    .trim();
}

function swapForSymbols(text) {
  return text
    .replace(/\band\b/gi, "&")
    .replace(/\bplus\b/gi, "+")
    .replace(/\bminus\b/gi, "-")
    .replace(/\bequals\b/gi, "=")
    .replace(/\bgreater than\b/gi, ">")
    .replace(/\bless than\b/gi, "<")
    .replace(/\bpercent\b/gi, "%")
    .replace(/\bnumber\b/gi, "#")
    .replace(/\bwith\b/gi, "w/")
    .replace(/\bwithout\b/gi, "w/o")
    .replace(/\bbecause\b/gi, "b/c");
}

function normalizeWhitespace(text) {
  let result = text;
  result = result.replace(/\s+/g, " ");
  result = result.replace(/\s+([.,!?;:])/g, "$1");
  result = result.replace(/([.,!?;:])\s*([.,!?;:])/g, "$1");
  result = result.replace(/\s*[-–]\s*/g, "-");
  result = result.replace(/\s*\/\s*/g, "/");
  result = result.replace(/\s*\|\s*/g, "|");
  result = result.trim();
  if (result && !/[.!?…]$/.test(result)) {
    result += ".";
  }
  return result;
}

function dedupeRepeatedSegments(text) {
  const seen = new Set();
  const parts = [];
  let start = 0;

  const pushTextChunk = (chunk) => {
    if (!chunk) return;
    const normalized = chunk.trim().toLowerCase();
    if (!normalized) return;
    if (seen.has(normalized)) return;
    seen.add(normalized);
    parts.push(chunk);
  };

  const pushNewlineChunk = (chunk) => {
    if (!chunk) return;
    if (parts.length === 0) return;
    const last = parts[parts.length - 1];
    if (last && /^\n+$/.test(last)) {
      if (chunk.length > last.length) {
        parts[parts.length - 1] = chunk;
      }
      return;
    }
    parts.push(chunk);
  };

  for (let index = 0; index < text.length; index += 1) {
    const char = text[index];

    if (char === "\n") {
      if (start < index) {
        pushTextChunk(text.slice(start, index));
      }
      let newlineEnd = index + 1;
      while (newlineEnd < text.length && text[newlineEnd] === "\n") {
        newlineEnd += 1;
      }
      pushNewlineChunk(text.slice(index, newlineEnd));
      index = newlineEnd - 1;
      start = newlineEnd;
      continue;
    }

    if (/[.!?;:]/.test(char)) {
      pushTextChunk(text.slice(start, index + 1));
      start = index + 1;
    }
  }

  if (start < text.length) {
    pushTextChunk(text.slice(start));
  }

  let result = parts.join("");
  result = result.replace(/\n+$/, "");
  return result;
}

const compressionMethods = [
  {
    key: "preamble",
    label: "Shorten prompt preamble",
    description: "Condense common role descriptions at the top of the prompt.",
    apply: compressPromptPreamble,
  },
  {
    key: "phrases",
    label: "Simplify verbose phrases",
    description: "Swap multi-word fillers with concise alternatives.",
    apply: applyVerbosePhraseCompression,
  },
  {
    key: "abbreviations",
    label: "Use common abbreviations",
    description: "Replace long technical nouns with standard shorthand.",
    apply: applyAbbreviationCompression,
  },
  {
    key: "filler",
    label: "Remove filler words",
    description: "Drop repeated intensifiers while keeping the first mention for tone.",
    apply: removeRepeatedFillerWords,
  },
  {
    key: "articles",
    label: "Remove redundant articles",
    description: "Keep at most one article per sentence for clarity.",
    apply: trimRedundantArticles,
  },
  {
    key: "imperative",
    label: "Prefer imperative tone",
    description: "Cut optional language so instructions read as direct actions.",
    apply: preferImperativeTone,
  },
  {
    key: "symbols",
    label: "Use symbolic shorthand",
    description: "Swap arithmetic and comparison words for symbols where safe.",
    apply: swapForSymbols,
  },
  {
    key: "quantities",
    label: "Shorten numeric quantities",
    description: "Convert spelled-out numbers and units into compact numerals.",
    apply: compressQuantities,
  },
  {
    key: "dedupe",
    label: "Remove duplicate instructions",
    description: "Drop repeated sentences and paragraphs so only the first copy stays.",
    apply: dedupeRepeatedSegments,
  },
  {
    key: "cleanup",
    label: "Normalize whitespace & punctuation",
    description: "Tighten spacing and ensure the prompt ends cleanly.",
    apply: normalizeWhitespace,
  },
];

const defaultEnabledMethodKeys = compressionMethods
  .filter((method) => method.key !== "dedupe")
  .map((method) => method.key);

function compressPrompt(input, enabledKeys = defaultEnabledMethodKeys) {
  if (!input.trim()) return "";
  const keys = new Set(enabledKeys);
  let output = input;
  for (const method of compressionMethods) {
    if (keys.has(method.key)) {
      output = method.apply(output);
    }
  }
  return output.trim();
}
