// Pingbird Prompt Compressor - Extension UI Logic

let messageTimeout = null;

// State management
const state = {
  input: '',
  output: '',
  methodState: {},
};

// Initialize method state (all except 'dedupe' enabled by default)
compressionMethods.forEach((method) => {
  state.methodState[method.key] = method.key !== 'dedupe';
});

// DOM elements
const elements = {
  input: document.getElementById('input'),
  output: document.getElementById('output'),
  inputCount: document.getElementById('input-count'),
  outputCount: document.getElementById('output-count'),
  status: document.getElementById('status'),
  statSaved: document.getElementById('stat-saved'),
  statRate: document.getElementById('stat-rate'),
  compressBtn: document.getElementById('compress-btn'),
  copyBtn: document.getElementById('copy-btn'),
  clearBtn: document.getElementById('clear-btn'),
  selectAllBtn: document.getElementById('select-all-btn'),
  clearAllBtn: document.getElementById('clear-all-btn'),
  methodList: document.getElementById('method-list'),
  methodsCount: document.getElementById('methods-count'),
};

// Calculate stats
function calculateStats() {
  const originalLength = state.input.length;
  const optimizedLength = state.output.length;
  const saved = Math.max(0, originalLength - optimizedLength);
  const compressionRate = originalLength > 0 ? Math.round((saved / originalLength) * 100) : 0;

  return { originalLength, optimizedLength, saved, compressionRate };
}

// Update UI
function updateUI() {
  const stats = calculateStats();

  elements.inputCount.textContent = `${stats.originalLength} characters`;
  elements.outputCount.textContent = `${stats.optimizedLength} characters`;
  elements.statSaved.textContent = stats.saved;
  elements.statRate.textContent = `${stats.compressionRate}%`;

  updateMethodsCount();
}

// Update methods count
function updateMethodsCount() {
  const activeCount = Object.values(state.methodState).filter(Boolean).length;
  const totalCount = compressionMethods.length;

  if (activeCount === totalCount) {
    elements.methodsCount.textContent = 'All methods active';
  } else if (activeCount === 0) {
    elements.methodsCount.textContent = 'No methods selected';
  } else {
    elements.methodsCount.textContent = `${activeCount} of ${totalCount} active`;
  }
}

// Update status message
function updateStatus(message) {
  elements.status.textContent = message;

  if (messageTimeout) {
    clearTimeout(messageTimeout);
  }

  messageTimeout = setTimeout(() => {
    elements.status.textContent = 'Paste a prompt to get started.';
  }, 3500);
}

// Render method toggles
function renderMethods() {
  elements.methodList.innerHTML = '';

  compressionMethods.forEach((method) => {
    const item = document.createElement('div');
    item.className = 'method-item';

    const toggle = document.createElement('div');
    toggle.className = 'method-toggle';
    if (state.methodState[method.key]) {
      toggle.classList.add('active');
    }
    toggle.setAttribute('role', 'switch');
    toggle.setAttribute('aria-checked', state.methodState[method.key]);
    toggle.setAttribute('tabindex', '0');

    const text = document.createElement('span');
    text.className = 'method-text';

    const title = document.createElement('span');
    title.className = 'method-title';
    title.textContent = method.label;

    const description = document.createElement('span');
    description.className = 'method-description';
    description.textContent = method.description;

    text.appendChild(title);
    text.appendChild(description);

    item.appendChild(toggle);
    item.appendChild(text);

    // Click handler
    const toggleMethod = () => {
      state.methodState[method.key] = !state.methodState[method.key];
      toggle.classList.toggle('active');
      toggle.setAttribute('aria-checked', state.methodState[method.key]);
      updateMethodsCount();
    };

    toggle.addEventListener('click', toggleMethod);
    toggle.addEventListener('keydown', (e) => {
      if (e.key === 'Enter' || e.key === ' ') {
        e.preventDefault();
        toggleMethod();
      }
    });

    elements.methodList.appendChild(item);
  });
}

// Compress handler
function handleCompress() {
  if (!state.input.trim()) {
    updateStatus('Enter a prompt before running compression.');
    state.output = '';
    elements.output.value = '';
    updateUI();
    return;
  }

  const enabledKeys = Object.keys(state.methodState).filter((key) => state.methodState[key]);
  state.output = compressPrompt(state.input, enabledKeys);
  elements.output.value = state.output;
  updateUI();
  updateStatus('Prompt compressed. Review the result before using it.');
}

// Copy handler
async function handleCopy() {
  if (!state.output.trim()) {
    updateStatus('Nothing to copy yet - compress a prompt first.');
    return;
  }

  try {
    if (navigator.clipboard && navigator.clipboard.writeText) {
      await navigator.clipboard.writeText(state.output);
    } else {
      // Fallback for older browsers
      const textarea = document.createElement('textarea');
      textarea.value = state.output;
      textarea.setAttribute('readonly', '');
      textarea.style.position = 'absolute';
      textarea.style.left = '-9999px';
      document.body.appendChild(textarea);
      textarea.select();
      document.execCommand('copy');
      document.body.removeChild(textarea);
    }
    updateStatus('Copied the optimized prompt to your clipboard.');
  } catch (error) {
    console.error('[prompt-compressor] clipboard copy failed', error);
    updateStatus("Couldn't access the clipboard. Copy manually if needed.");
  }
}

// Clear handler
function handleClear() {
  state.input = '';
  state.output = '';
  elements.input.value = '';
  elements.output.value = '';
  updateUI();
  updateStatus('Cleared the workspace.');
}

// Select all methods
function handleSelectAll() {
  compressionMethods.forEach((method) => {
    state.methodState[method.key] = true;
  });
  renderMethods();
}

// Clear all methods
function handleClearAll() {
  compressionMethods.forEach((method) => {
    state.methodState[method.key] = false;
  });
  renderMethods();
}

// Event listeners
elements.input.addEventListener('input', (e) => {
  state.input = e.target.value;
  updateUI();
});

elements.compressBtn.addEventListener('click', handleCompress);
elements.copyBtn.addEventListener('click', handleCopy);
elements.clearBtn.addEventListener('click', handleClear);
elements.selectAllBtn.addEventListener('click', handleSelectAll);
elements.clearAllBtn.addEventListener('click', handleClearAll);

// Keyboard shortcuts
elements.input.addEventListener('keydown', (e) => {
  if ((e.ctrlKey || e.metaKey) && e.key === 'Enter') {
    e.preventDefault();
    handleCompress();
  }
});

// Installation tracking
async function trackInstallation() {
  try {
    // Get or create install ID
    let installId = localStorage.getItem('pingbird_install_id');
    if (!installId) {
      // Generate unique install ID
      installId = `ext_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
      localStorage.setItem('pingbird_install_id', installId);
    }

    // Detect browser info
    const userAgent = navigator.userAgent;
    let browser = 'unknown';
    let browserVersion = '';

    if (userAgent.includes('Firefox/')) {
      browser = 'firefox';
      const match = userAgent.match(/Firefox\/([\d.]+)/);
      if (match) browserVersion = match[1];
    } else if (userAgent.includes('Edg/')) {
      browser = 'edge';
      const match = userAgent.match(/Edg\/([\d.]+)/);
      if (match) browserVersion = match[1];
    } else if (userAgent.includes('Chrome/')) {
      browser = 'chrome';
      const match = userAgent.match(/Chrome\/([\d.]+)/);
      if (match) browserVersion = match[1];
    } else if (userAgent.includes('Safari/')) {
      browser = 'safari';
      const match = userAgent.match(/Version\/([\d.]+)/);
      if (match) browserVersion = match[1];
    }

    // Detect OS
    let os = 'unknown';
    let osVersion = '';

    if (userAgent.includes('Win')) {
      os = 'Windows';
      const match = userAgent.match(/Windows NT ([\d.]+)/);
      if (match) osVersion = match[1];
    } else if (userAgent.includes('Mac')) {
      os = 'macOS';
      const match = userAgent.match(/Mac OS X ([\d_]+)/);
      if (match) osVersion = match[1].replace(/_/g, '.');
    } else if (userAgent.includes('Linux')) {
      os = 'Linux';
    }

    // Track installation/usage
    await fetch('https://www.pingbird.xyz/api/extension/install', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        installId,
        browser,
        browserVersion,
        extensionVersion: '1.0.0',
        os,
        osVersion,
        language: navigator.language,
        timezone: Intl.DateTimeFormat().resolvedOptions().timeZone,
        screenResolution: `${screen.width}x${screen.height}`,
      }),
    });
  } catch (error) {
    // Silently fail - don't break extension functionality
    console.error('Installation tracking failed:', error);
  }
}

// Initialize
renderMethods();
updateUI();
trackInstallation();
